/******************************************************************************
*  SCT_L - is used for center aligned PWM at SCT_OUT0
*
*************************  SCT_L half  ****************************************
*
*       |       PWM0 Period       |                         |
*       |                         |                         |
*                 +-----+                   +-----+                   +-----+
*                 |     |                   |     |                   |     |      OUT0
*    ---+---------+     +---------+---------+     +---------+---------+     +----
*
*                 |     |                   |     |                   |
*                EV0   EV0                 EV0   EV0                 EV0
*
*  MATCH0_L used for PWM0 frequency period
*  EV0 - all states - on MATCH1_L - set OUT0 if up counting, clear OUT0 if down counting
*
*  P0.14 [O] - SCT_OUT0 : PWM
*
******************************************************************************/
#include "LPC8xx.h"

#define PWM             14                                  // PWM at port pin P0_14
#define OUT0            0                                   // SCT_OUT0 as PWM output
#define PWM_FREQ        10000                               // PWM required frequency = 10KHz
#define PWM_PERIOD      (SystemCoreClock / (PWM_FREQ * 2))  // PWM counter period (*2 because of bi-dir mode)
                                                            // example 24MHz/10KHz*2 = 1200 SCT clocks

void PWM_set(uint8_t val)                                   // set PWM duty cycle (from 0 to 100%)
{
#define PWM_STEP        (PWM_PERIOD / 100)                  // PWM resolution in 100 steps

    if (val == 0)                                           // check val between 0% and 100%
       LPC_SCT->MATCHREL[1].L = 0;
    else if (val < 100)
       LPC_SCT->MATCHREL[1].L = (PWM_STEP * val) - 1;
    else
       LPC_SCT->MATCHREL[1].L = PWM_PERIOD - 2;             // set to 100% duty cycle
}

void SCT_Init(void)
{
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 7) | (1 << 8);       // enable the SWM and SCT clock

    LPC_SWM->PINASSIGN6 &= ((PWM << 24) | 0x00FFFFFF);      // SCT_OUT0 = PWM

    LPC_SCT->CONFIG           |= (1 << 17);                 // auto limit _L (on match 0)
    LPC_SCT->OUTPUT           |= (0 << OUT0);               // preset OUT0 low
//  LPC_SCT->OUTPUT           |= (1 << OUT0);               // preset OUT0 high (for low active signal)
    LPC_SCT->OUTPUTDIRCTRL    |= (0x1 << 0);                // reverse OUT0 set/clr when counter _L is
                                                            // down counting (center aligned mode)
    LPC_SCT->CTRL_L           |= (1 << 4);                  // bi-dir count mode

    LPC_SCT->MATCH[0].L        = PWM_PERIOD - 1;            // match 0 @ PWM freq
    LPC_SCT->MATCH[1].L        = 0;                         // use match 1 for PWM duty cycle

    LPC_SCT->EVENT[0].STATE    = 0x00000003;                // event 0 happens in all states (both 0 and 1)
    LPC_SCT->EVENT[0].CTRL     = (1 << 0)  |                // MATCHSEL[3:0]   = related to match 1
                                 (0 << 4)  |                // HEVENT[4]       = event 0 belongs to the L timer
                                 (1 << 12) |                // COMBMODE[13:12] = match condition only
                                 (0 << 14) |                // STATELD[14]     = STATEV is added to state
                                 (0 << 15);                 // STATEV[15]      = 0 (no change)

    LPC_SCT->OUT[OUT0].SET     = (1 << 0);                  // event 0 sets the OUT0 signal
//  LPC_SCT->OUT[OUT0].CLR     = (1 << 0);                  // event 0 clears the OUT0 signal (for low active mode)

    LPC_SCT->CTRL_L           &= ~(1 << 2);                 // start the _L counter
}
